<?php

namespace YayMailAddonConditionalLogic\Services;

use YayMailAddonConditionalLogic\SingletonTrait;
defined( 'ABSPATH' ) || exit;

/**
 * ConditionalLogicService Class
 *
 * @method static ConditionalLogicService get_instance()
 */
class ConditionalLogicService {
    use SingletonTrait;

    private $wc_email_args;

    private $logic_functions_map = [
        'billing_country'    => 'check_billing_country',
        'product_categories' => 'check_product_category',
        'products'           => 'check_product_ids',
        'product_skus'       => 'check_product_skus',
        'order_min'          => 'check_order_min',
        'order_max'          => 'check_order_max',
        'payment_methods'    => 'check_payment_method',
        'shipping_methods'   => 'check_shipping_method',
        'coupon_codes'       => 'check_coupon_codes',
        'payment_status'     => 'check_payment_status',
    ];

    /**
     * Check if the element meets the conditions
     *
     * @var array $conditions The conditions array
     * @var String $condition_type Initial value for the $carry, could either be 'all' or 'any
     * @var object $wc_email_args
     * @return boolean Return true if the element meets the conditions
     */
    public function meets_conditional_logics( $conditions, $condition_type, $wc_email_args ) {
        if ( empty( $conditions ) ) {
            return true;
        }

        $this->wc_email_args = apply_filters( 'yaymail_conditional_logic_prepared_wc_email_args', $wc_email_args, $conditions, $condition_type );

        $is_showed = array_reduce(
            $conditions,
            function ( $carry, $condition ) use ( $condition_type ) {
                if ( empty( $condition ) ) {
                    return $carry;
                }
                $result = $this->check_condition( $condition );
                return ( 'any' === $condition_type ) ? ( $carry || $result ) : ( $carry && $result );
            },
            'all' === $condition_type
        );

        return $is_showed;
    }

    /**
     * Check a specific condition based on the provided logic.
     *
     * @param array $condition An array specifying the condition to check. It should have the following structure:
     *   - comparing_operator: The operator used for comparison (e.g., "excludesAll").
     *   - logic: The type of condition to check (e.g., "billing_country").
     *   - name: A human-readable name for the condition (e.g., "Billing country").
     *   - value: An array containing data relevant to the condition (specific to the logic).
     *
     * @return boolean Returns the result of the condition check.
     * True if the element passed the condition
     */
    private function check_condition( $condition ) {
        $logic = $condition['logic'] ?? '';

        $order = $this->wc_email_args['order'] ?? null;

        // Avoid processing if the order is 'SampleOrder'
        if ( ! $order || 'SampleOrder' === $order ) {
            return false;
        }

        $condition['value'] = is_array( $condition['value'] ) ? array_map(
            function( $value ) {
                return $value['value'] ?? $value;
            },
            $condition['value']
        ) : $condition['value'];

        $check = false;
        // Execute the logic function if it exists
        if ( isset( $this->logic_functions_map[ $logic ] ) ) {
            $function_name = $this->logic_functions_map[ $logic ];
            $check         = $this->$function_name( $condition );
        }

        return apply_filters( 'yaymail_conditional_logic_condition_result', $check, $condition, $this->wc_email_args, $this->logic_functions_map );
    }

    /**
     * Get the final result by applying a comparing operator to two arrays.
     *
     * - 'is': Returns true if at least one element in the needle_array appears in the hay_array.
     * - 'isNot': Returns true if none of the elements in the needle_array appear in the hay_array.
     * - 'containsAll': Returns true if all elements in the needle_array are present in the hay_array.
     *
     * @param array|string               $needles An array containing elements to be compared.
     * @param array|string               $haystack An array in which elements are compared against the needle_array.
     * @param 'is'|'isNot'|'containsAll' $comparing_operator The comparing operator to determine the result.
     *
     * @return bool The result of the comparison based on the chosen operator.
     */
    private function get_result_by_applying_comparing_operator( $needles, $haystack, $comparing_operator ) {

        $needles  = (array) $needles;
        $haystack = (array) $haystack;

        switch ( $comparing_operator ) {
            case 'is':
                /**
                 * Return true when at least 1 needle appears in haystack
                 */
                return ! empty( array_intersect( $needles, $haystack ) );
            case 'isNot':
                /**
                 * Return true when none of the needles appear in haystack
                 */
                return empty( array_intersect( $needles, $haystack ) );
            case 'containsAll':
                /**
                 * Return true when all needles appear in haystack
                 */
                return empty( array_diff( $needles, $haystack ) );
            default:
                return false;
        }
    }

    private function check_billing_country( $condition ) {
        $condition_value_countries = $condition['value'];

        $order_billing_country = $this->wc_email_args['order']->get_billing_country();

        return $this->get_result_by_applying_comparing_operator( $condition_value_countries, $order_billing_country, $condition['comparing_operator'] );
    }

    private function check_product_category( $condition ) {
        $condition_value_category_ids = $condition['value'] ?? null;

        $items = $this->wc_email_args['order']->get_items();

        $order_product_category_ids = [];
        foreach ( $items as $key => $item ) {
            $product_id = $item['product_id'];
            $terms      = get_the_terms( $product_id, 'product_cat' );
            foreach ( $terms as $term ) {
                if ( $term->parent ) {
                    $parent_term                  = get_term( $term->parent );
                    $order_product_category_ids[] = $parent_term->slug;
                }
                $order_product_category_ids[] = $term->slug;
                // In old yaymail, this is term->id
            }
        }

        return $this->get_result_by_applying_comparing_operator( $condition_value_category_ids, $order_product_category_ids, $condition['comparing_operator'] );
    }

    private function check_order_min( $condition ) {
        $condition_value_order_min = $condition['value'];
        $order_total               = $this->wc_email_args['order']->get_total();
        return $condition_value_order_min <= $order_total;
    }

    private function check_order_max( $condition ) {
        $condition_value_order_max = $condition['value'];
        $order_total               = $this->wc_email_args['order']->get_total();
        return $condition_value_order_max >= $order_total;
    }

    private function check_payment_method( $condition ) {
        $condition_value_payment_methods = $condition['value'];
        $order_payment_method            = $this->wc_email_args['order']->get_payment_method();

        return $this->get_result_by_applying_comparing_operator( $condition_value_payment_methods, $order_payment_method, $condition['comparing_operator'] );
    }

    private function check_shipping_method( $condition ) {
        $condition_value_shipping_methods = $condition['value'];

        $order_shipping_method_ids = [];
        foreach ( $this->wc_email_args['order']->get_items( 'shipping' ) as $item_id => $item ) {
            $item_data                   = $item->get_data();
            $order_shipping_method_ids[] = $item_data['method_id'];
        }

        return $this->get_result_by_applying_comparing_operator( $condition_value_shipping_methods, $order_shipping_method_ids, $condition['comparing_operator'] );
    }

    private function check_coupon_codes( $condition ) {
        $condition_value_coupon_codes = array_map( 'strtolower', $condition['value'] );

        $order_coupon_codes = array_map( 'strtolower', $this->wc_email_args['order']->get_coupon_codes() );

        return $this->get_result_by_applying_comparing_operator( $condition_value_coupon_codes, $order_coupon_codes, $condition['comparing_operator'] );
    }

    private function check_product_ids( $condition ) {
        $condition_value_products_ids = $condition['value'];

        $order_line_items = $this->wc_email_args['order']->get_data()['line_items'];

        $order_product_ids = array_values(
            array_map(
                function( $item ) {
                    if ( isset( $item->get_data()['product_id'] ) ) {
                        return $item->get_data()['product_id'];
                    } else {
                        return -1;
                    }
                },
                $order_line_items
            )
        );

        return $this->get_result_by_applying_comparing_operator( $condition_value_products_ids, $order_product_ids, $condition['comparing_operator'] );
    }

    private function check_product_skus( $condition ) {
        $line_items = $this->wc_email_args['order']->get_data()['line_items'];

        $order_product_skus = [];
        if ( $line_items ) {
            foreach ( $line_items as $key => $line_item ) {
                if ( ! isset( $line_item->get_data()['product_id'] ) ) {
                    continue;
                }

                if ( isset( $line_item->get_data()['variation_id'] ) && $line_item->get_data()['variation_id'] !== 0 ) {
                    $variation_id = $line_item->get_data()['variation_id'];
                    $sku_meta     = get_post_meta( $variation_id, '_sku', true );
                }
                if ( ! isset( $sku_meta ) || empty( $sku_meta ) ) {
                    $product_id = $line_item->get_data()['product_id'];
                    $sku_meta   = get_post_meta( $product_id, '_sku', true );
                }
                $order_product_skus[] = $sku_meta;
                unset( $sku_meta );
            }
        }

        return $this->get_result_by_applying_comparing_operator( $condition['value'], $order_product_skus, $condition['comparing_operator'] );
    }

    private function check_payment_status( $condition ) {

        // if ( ! empty( $this->wc_email_args['booking'] ) && ! empty( $this->wc_email_args['booking']->get_data()['order_id'] ) ) {
        // Booking Order
        // $order = wc_get_order( $this->wc_email_args['booking']->get_data()['order_id'] );
        // } elseif ( ! empty( $wc_email_args['order'] ) ) {
        // Normal Order
        // $order = $wc_email_args['order'];
        // }

        $order = $this->wc_email_args['order'];

        $condition_value_payment_statuses = $condition['value'];
        if ( empty( $condition_value_payment_statuses ) ) {
            return false;
        }

        // Map all values to lowercase
        $lowercased_condition_value_payment_statuses = array_map( 'strtolower', $condition_value_payment_statuses );

        $order_status = $order->get_data()['status'];

        if ( $order_status !== 'pending' ) {
            $order_status = 'processing';
        }

        return $this->get_result_by_applying_comparing_operator( $lowercased_condition_value_payment_statuses, $order_status, $condition['comparing_operator'] );
    }
}
